// File: chess3d.cpp
//
// Description: A program to play 3D chess. Uncomment
//              the '#define TEXT_MODE' statement at the
//              top of the listing to produce a "text only"
//              version of the program (useful for debugging).
//
//       No binary rights reserved: this software may be
//       used in compiled form without restrictions.  All
//       source rights reserved: Source code to the GUI
//       library can not be distributed (on bulletin boards, 
//       or as part of shareware or commercial products)
//       without written permission.
//
//       This software is provided "as is". The user accepts 
//       all responsibility for its use.
//


// #define TEXT_MODE

#include "applib.h"
#include <iostream.h>
#include <fstream.h>

const int MAX_DEPTH = 4;
const int MAX_MOVES = 220;

const int WHITE = 1;
const int BLACK =-1;

const int PAWN = 1;
const int KNIGHT = 2;
const int BISHOP = 3;
const int ROOK = 4;
const int QUEEN = 5;
const int KING = 6;

//
// Conversion from [row, column, level] indices
// to linear index:
//

int get_index(int row, int col, int level)
{
    return 22+col+row*10+level*120;
}

//
// Conversion from a linear board array index to
// [row, column, level] indices:
//

void index_to_coords(int index, int &row,
                     int &col, int &level)
{
    level = index / 120;
    int coord_in_level = index - 120 * level;
    row = (coord_in_level/10) - 2;
    col = coord_in_level - (row+2)*10 - 2;
}

class board {
 public:
    board();
    int move_from[MAX_MOVES]; // "from squares" for
                              // legal moves
    int move_to[MAX_MOVES];   // "to squares" for
                              // legal moves
    int move_value[MAX_MOVES];  // static evaluation
                                // of this move
    int num_possible_moves;
    void set_pieces();  // beginning of game setup
#ifdef TEXT_MODE
    void print_board();
    void print_possible_moves();
#endif
    // defines move_to and move_from:
    void generate_moves(int protect);
    int generate_moves_from_square(int index, int protect);
    board & operator = (board &b);
    int * get_board() { return brd; }
    int * get_white_control_address() { return wcontrol; }
    int * get_black_control_address() { return bcontrol; }
    int get_current_side() { return current_side; }
    void set_current_side(int side) { current_side = side;}

 private:
    int brd[360];
    int bcontrol[360];
    int wcontrol[360];
    int current_side;  // toggles between -1 and 
                       // 1 for lookahead
};

class game {
 public:
    game(int look_ahead_depth, int search_width);
    int depth, width;
    // use a static evaluation function to impose a
    // partial ordering on the moves:
    int static_evaluation(int brd_level, 
                          int possible_move_index);
    void rank_moves(int level);
    int make_best_move(int level, int side_flag);
    board * get_board_address(int index)
    { return &(board_stack[index]); }
 private:
    board board_stack[MAX_DEPTH]; // board_stack[0] holds
                                  // the current position
};

//
// Class constructor for 'board':
//

board::board()
{
    num_possible_moves = 0;
    for (int i=0; i<360; i++)
        brd[i] = bcontrol[i] = wcontrol[i] = 0;
}

//
// Define the = operator for board:
//

board & board::operator = (board &b)
{
    for (int i=0; i<360; i++) {
        brd[i] = b.brd[i];
        bcontrol[i] = b.bcontrol[i];
        wcontrol[i] = b.wcontrol[i];
    }
    current_side = b.current_side;
    num_possible_moves = b.num_possible_moves;
    for (i=0; i<num_possible_moves; i++) {
        move_from[i] = b.move_from[i];
        move_to[i] = b.move_to[i];
        move_value[i] = b.move_value[i];
    }
    return (*this); 
}

//
// Set up the initial board position:
//

void board::set_pieces()
{
    for (int i=0; i<360; i++) // zero out board squares
    {
        brd[i] = 0;
    }
    for (i=0; i<12; i++) // off brd squares
                         // (see Figure 17.2)
    {
        brd[10*i] = brd[10*i+120] = brd[10*i+240] = 7;
        brd[10*i+1] = brd[10*i+121] = brd[10*i+241] = 7;
    }
    for (i=2; i< 10; i++) // off brd squares
                          // (see Figure 17.2)
    {
        brd[i] = brd[i+120] = brd[i+240] = 7;
        brd[i+10] = brd[i+130] = brd[i+250] = 7;
        brd[i+100] = brd[i+220] = brd[i+320] = 7;
        brd[i+110] = brd[i+230] = brd[i+350] = 7;
    }
    
    for (i=32; i<40; i++)
         brd[i] = WHITE * PAWN;
    brd[22] = WHITE * ROOK;    
    brd[23] = WHITE * KNIGHT;
    brd[24] = WHITE * BISHOP;
    brd[25] = WHITE * QUEEN;
    brd[26] = WHITE * KING;
    brd[27] = WHITE * BISHOP;
    brd[28] = WHITE * KNIGHT; 
    brd[29] = WHITE * ROOK;

    for (i=82+240; i<90+240; i++) 
         brd[i] = BLACK * PAWN;
    brd[92+240] = BLACK * ROOK;
    brd[93+240] = BLACK * KNIGHT;
    brd[94+240] = BLACK * BISHOP;
    brd[95+240] = BLACK * QUEEN;
    brd[96+240] = BLACK * KING;
    brd[97+240] = BLACK * BISHOP;
    brd[98+240] = BLACK * KNIGHT;
    brd[99+240] = BLACK * ROOK;
}

#ifdef TEXT_MODE
// Print out the board to stderr:
void board::print_board()
{
    // loop over squares on all three levels:
    for (int i=99+2*120; i>=0; i-=10) 
    {
        if (i==99+2*120) cerr << "\n\nLevel 3:\n";
        if (i==99+120) cerr <<   "\nLevel 2:\n";
        if (i== 99) cerr <<   "\nLevel 1:\n";
        for (int j=0; j<8; j++) {
            int ind = i - 7 + j;
            if (brd[ind] != 7) { // not off of the board:
                if (brd[ind] == 0) {
                    int index = ind % 120;
                    int row = (index / 10) % 2;
                    int col = (index - ((index / 10) * 10))
                             % 2;
                    if (row == 0) {
                        if (col == 0) cerr << " X ";
                         else         cerr << " . ";
                    } else {
                        if (col == 1) cerr << ".X.";
                         else         cerr << " . ";
                    }
                } else {
                    if (brd[ind] < 0) cerr << " B";
                     else             cerr << " W";
                    int ptype = brd[ind];
                    if (ptype < 0) ptype = -ptype;
                    if (ptype == PAWN) cerr << "P";
                    if (ptype == KNIGHT) cerr << "N";
                    if (ptype == BISHOP) cerr << "B";
                    if (ptype == ROOK) cerr << "R";
                    if (ptype == QUEEN) cerr << "Q";
                    if (ptype == KING) cerr << "K";
                }
            }
        }
        if (brd[i] != 7)  cerr << "\n";
    }
    cerr << "\n";
}

void board::print_possible_moves()
{   int row1, col1, level1, row2, col2, level2;
    for (int i=0; i<num_possible_moves; i++) 
    {
        index_to_coords(move_from[i],
                        row1,
                        col1,
                        level1);
        index_to_coords(move_to[i],
                        row2,
                        col2,
                        level2);
        cerr << "From level " << level1 
             << ", row " << row1 << ", col " 
             << col1 << "  to: level " 
             << level2 <<", row " << row2 
             << ", col " << col2 << "  value: " 
             << move_value[i] << "\n";
    }
}
#endif

//
// The following member function defines the data arrays
// move_to and move_from generating possible moves:
//

void board::generate_moves(int protect)
{
    num_possible_moves = 0;
    for (int i=0; i<360; i++)
        if (brd[i] != 7 && brd[i] != 0) {
            if (current_side == -1 && brd[i] < 0)
                generate_moves_from_square(i, protect);
            if (current_side == 1  && brd[i] > 0)
                generate_moves_from_square(i, protect);
        }
    return;
}


static int index_table[] = {0, 12, 15, 10, 1, 6, 6};
static int piece_table[] = {0, 1, -1, 10, -10, 0, 1, -1, 10,
                            -10, -9, -11, 9, 11, 0, 8, -8,
                            12, -12, 19, -19, 21, -21, 0, 10,
                            20, 0, 0, 0, 0, 0};
static int control[] = {0, 12, 10, 10, 9, 8, 8};

int board::generate_moves_from_square(int index, int protect)
{   int k;
    int level = index / 120;
    int ind = index - 120*level;  // index inside a level
    int num_moves = 0;
    int table;
    int row;
    int move_index;
    int table_loop;
    int ptype = brd[index]; if (ptype < 0) ptype = -ptype;
    
    // Regardless of piece type, allow movement up or down
    // a level to either a blank square or for a capture:
    if (level < 2) // check for moving up a level:
    {
        if (brd[index] > 0) // white piece
        {
            if (brd[index+120] <= 0)
            {
                move_from[num_possible_moves] = index;
                move_to[num_possible_moves++] = index + 120;
                if (current_side==1)
                    wcontrol[index + 120]+=control[ptype];
                  else 
                    bcontrol[index + 120]+=control[ptype];
            }
        }
        if (brd[index] < 0) // black piece
        {
            if (brd[index+120] >= 0) 
            {
                move_from[num_possible_moves] = index;
                move_to[num_possible_moves++] = index + 120;
                if (current_side==1)
                    wcontrol[index + 120]+=control[ptype];
                  else
                    bcontrol[index + 120]+=control[ptype];
            }
        }
    }

    if (level > 0) // check for moving down a level:
    {
        if (brd[index] > 0) // white piece
        {
            if (brd[index-120] <= 0) 
            {
                move_from[num_possible_moves] = index;
                move_to[num_possible_moves++] = index - 120;
                if (current_side==1)
                    wcontrol[index - 120]+=control[ptype];
                  else
                    bcontrol[index - 120]+=control[ptype];
            }
        }
        if (brd[index] < 0) // black piece
        {
            if (brd[index-120] >= 0) 
            {
                move_from[num_possible_moves] = index;
                move_to[num_possible_moves++] = index - 120;
                if (current_side==1)
                    wcontrol[index - 120]+=control[ptype];
                  else
                    bcontrol[index - 120]+=control[ptype];
            }
        }
    }
                        
    switch (ptype) {
        case PAWN:  // There are MANY special cases for pawns:
        row = (ind / 10) - 2; // index rows starting with 0
            if (current_side == BLACK) 
            {
                if (row == 6 && level == 2) // pawn has not
                                            // moved yet:
                {
                    if (brd[index-20]==0) 
                    {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = index-20;
                        if (brd[index-20-120] <= 0) // empty or 
                                                    // capture OK
                        {
                            move_from[num_possible_moves] = index;
                            move_to[num_possible_moves++] =
                                index - 20 - 120;
                        }
                    }
        
                }
                if (brd[index-10]==0) // not blocked:
                {
                    move_from[num_possible_moves] = index;
                    move_to[num_possible_moves++] = index-10;
                    if (row==6 && level == 2) 
                    {
                        if (brd[index-10-120] <= 0) // empty or
                                                    // capture OK
                        {
                            move_from[num_possible_moves] = index;
                            move_to[num_possible_moves++] = 
                                index - 10 - 120;
                        }
                    }
                }
                if (brd[index-9] != 7 && brd[index-9]!=0) 
                {
                    if (brd[index-9] < 0) 
                    {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = index-9;
                    }
                }
                if (brd[index-11] != 7 && brd[index-11]!=0) 
                {
                    if (brd[index-11] < 0) 
                    {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = index-11;
                    }
                }                       
            } else { // WHITE
                if (row == 1) // pawn has not moved yet:
                {
                    if (brd[index+20]==0) 
                    {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = index+20;
                        if (brd[index+20+120] <= 0) { // empty or
                                                      // capture OK
                            move_from[num_possible_moves] = index;
                            move_to[num_possible_moves++] =
                                index + 20 + 120;
                        }
                    }
        
                }
                if (brd[index+10]==0) // not blocked:
                {
                    move_from[num_possible_moves] = index;
                    move_to[num_possible_moves++] = index+10;
                    if (row==1) 
                    {
                        if (brd[index+10+120] <= 0) // empty or
                        {                           // capture OK
                            move_from[num_possible_moves] = index;
                            move_to[num_possible_moves++] = 
                                index + 10 + 120;
                        }
                    }
                }
                if (brd[index+9] != 7 && brd[index+9]!=0)
                {
                    if (brd[index+9] < 0) 
                    {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = index+9;
                    }
                }
                if (brd[index+11] != 7 && brd[index+11]!=0) 
                {
                    if (brd[index+11] < 0) 
                    {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = index+11;
                    }
                }                       
            }
            // Fill in control arrays for all possible
            // pawn captures:
            for (k=22; k<360; k++) 
            {
                if (brd[k] == BLACK*PAWN) 
                {
                    bcontrol[k-9] += 12;
                    bcontrol[k-11] += 12;
                }
                if (brd[k] == WHITE*PAWN) 
                {
                    wcontrol[k+9] += 12;
                    wcontrol[k+11] += 12;
                }
            }
            break;
            
    case KNIGHT: // after pawn moves, the other piece
                 // moves are simple:
        case BISHOP:
        case ROOK:
        case QUEEN:
        case KING:
        move_index = index_table[ptype];
        for (table_loop=move_index; 
             piece_table[table_loop]!=0;
             table_loop++) 
        {
            if (ptype == KNIGHT || ptype == KING) 
            {
                if (brd[index + piece_table[table_loop]] != 7)  
                {
                    if (brd[index + piece_table[table_loop]] *
                        current_side <= 0) 
                    {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = 
                            index + piece_table[table_loop];
                        if (current_side==1)
                            wcontrol[
                             index + piece_table[table_loop]
                                    ]+=control[ptype];
                        else        
                            bcontrol[
                             index + piece_table[table_loop]
                                    ]+=control[ptype];
                    } else 
                    {
                        if (brd[index + piece_table[table_loop]] *
                            current_side > 0 &&
                            protect==1) 
                        {
                            if (current_side==1)
                               wcontrol[
                                index + piece_table[table_loop]
                                       ]+=control[ptype];
                            else       
                               bcontrol[
                                index + piece_table[table_loop]
                                       ]+=control[ptype];
                        }
                    }
                }
            }  else // BISHOP, ROOK, or QUEEN:
            {
                for (int sq=index+piece_table[table_loop];
                     ;
                     sq+=piece_table[table_loop])
                {
                    if (brd[sq] == 7)  break;  // off of board
                    if (brd[sq] == 0) {
                        move_from[num_possible_moves] = index;
                        move_to[num_possible_moves++] = sq;
                        if (current_side==1)
                            wcontrol[sq] =
                                 wcontrol[sq] + control[ptype];
                        else    
                            bcontrol[sq] = 
                                 bcontrol[sq] + control[ptype];

#ifdef TEXT_MODE
#ifdef DEBUG
                    cerr << "-- ptype=" << ptype 
                         << ", control[ptype]=" <<
                         control[ptype] << ", sq=" << sq
                         << ", current_side="
                         << current_side << "\n";
                    cerr << "++ wcontrol[35]= " <<
                         wcontrol[35] << "bcontrol[35]="
                         << bcontrol[35] << "\n";
#endif
#endif
                    }
                    if (brd[sq] != 0) 
                    {
                        int capture_sign = 1;
                        if (brd[sq] < 0) capture_sign = -1;
                        if (protect == 1)
                        {
                            if (capture_sign == current_side) 
                            {
                                if (current_side==1)
                                   wcontrol[sq]+=control[ptype];
                                else
                                   bcontrol[sq]+=control[ptype];
                                break; // protect friendly piece
                            }
                        }
                        if (capture_sign != current_side)
                        {
                            move_from[num_possible_moves] = index;
                            move_to[num_possible_moves++] = sq;
                            if (current_side==1)
                               wcontrol[sq]+=control[ptype];
                              else 
                               bcontrol[sq]+=control[ptype];
                            break; // capture
                        } else {
                            break;
                        }
                    }
                }       
            }
            if (num_possible_moves > 0)
                if (move_to[num_possible_moves-1] < 22  ||
                    move_to[num_possible_moves-1] > 340)
                    num_possible_moves--;
        }
        break;
    }
    return num_possible_moves;
}

//
// Constructor for class 'game':
//

game::game(int look_ahead_depth, int search_width)
{
    depth = look_ahead_depth; width = search_width;
    board_stack[0].set_pieces();
}

// Static evaluation function:

static int p_value[] = {0, 1, 3, 3, 5, 9, 1000};

const int LEVEL_MOVE_BONUS = 3;

int game::static_evaluation(int brd_level, 
                            int possible_move_index)
{
    int value = 0;
    
    if (brd_level > (MAX_DEPTH - 2)) {
#ifdef TEXT_MODE
        cerr << "Error in static evaluation\n";
#endif
        return 0;
    }

    // Copy board to next board stack level:
    board_stack[brd_level+1] = board_stack[brd_level];
    int *b = get_board_address(brd_level+1)->get_board();

    int to_square   = 
      board_stack[brd_level].move_to[possible_move_index];
    int from_square = 
      board_stack[brd_level].move_from[possible_move_index];
    
    b[to_square] = b[from_square];
    b[from_square] = 0;

    board *bb = get_board_address(brd_level+1);
    for (int i=0; i<360; i++) {
        bb->get_white_control_address()[i] = 0;
        bb->get_black_control_address()[i] = 0;
    }
    
    int side_moving =
       board_stack[brd_level+1].get_current_side();
    
    if (side_moving == BLACK) 
    {
        if ((to_square / 120) < (from_square / 120))
           value += LEVEL_MOVE_BONUS;
        if (b[to_square-9] == WHITE*PAWN ||
            b[to_square-11] == WHITE*PAWN)
        {
            value -= 50 * (p_value[-b[to_square]] - p_value[PAWN]);
        }
    }  else {  // side_moving == WHITE
        if ((to_square / 120) > (from_square / 120))
            value += LEVEL_MOVE_BONUS;
        if (b[to_square+9] == BLACK*PAWN ||
            b[to_square+11] == BLACK*PAWN)
        {
            value -= 50 * (p_value[b[to_square]] - p_value[PAWN]);
        }
    }
    
    if (b[to_square] == BLACK*QUEEN ||
        b[to_square] == WHITE*QUEEN)
    {
        value -= 2;
    }

    if (b[to_square] == BLACK*KING ||
        b[to_square] == WHITE*KING) 
    {
        value -= 3;
    }
    
    board_stack[brd_level+1].set_current_side(1);

    for (i=22; i<360; i++)
    {
        if (board_stack[brd_level+1].get_board()[i] != 7 &&
            board_stack[brd_level+1].get_board()[i] > 0)
        {
            board_stack[brd_level+1].generate_moves_from_square(i, 1);
        }
    }
    
#ifdef TEXT_MODE
#ifdef DEBUG
    cerr << "Number white moves= " 
         << board_stack[brd_level+1].num_possible_moves;
#endif
#endif
                
    board_stack[brd_level+1].set_current_side(-1);

    for (i=22; i<360; i++)
    {
        if (board_stack[brd_level+1].get_board()[i] != 7 &&
            board_stack[brd_level+1].get_board()[i] < 0)
        {
            board_stack[brd_level+1].
                       generate_moves_from_square(i, 1);
        }
    }
    
#ifdef TEXT_MODE
#ifdef DEBUG
    cerr << ", number black moves= "
         << board_stack[brd_level+1].num_possible_moves << "\n";
#endif
#endif

#if 0
    // We need to subtract 1 from the control array
    // for the square being moved to:
    if (side_moving==1)
        board_stack[brd_level+1].wcontrol[to_square] -=1;
     else
        board_stack[brd_level+1].bcontrol[to_square] -=1;
#endif

    // Static evaluation based on control and material:
    for (i=22; i<360; i++) 
    {
        if (b[i] != 7) 
        {
            int ptype = b[i]; if (ptype < 0)  ptype = - ptype;
            int pside = 1; if (b[i] < 0) pside = -1;
            int relative_control =
               board_stack[brd_level+1].
                  get_white_control_address()
                      [i]-board_stack[brd_level+1].
                      get_black_control_address()[i];
            value += side_moving * relative_control / 10;
            value += side_moving * 100 * p_value[ptype] * pside;
            if (b[i] != 0)
            {
                if (b[i] > 0 && relative_control < 0)
                    value -= side_moving * 15 * p_value[ptype];
                if (b[i] < 0 && relative_control > 0)
                    value += side_moving * 15 * p_value[ptype];
            }
            if (b[i] == WHITE * KING) 
            {
                if (side_moving == WHITE)
                    if (board_stack[brd_level+1].
                       get_black_control_address()[i]>0)
                       value-=1000; // king in check
                if (side_moving == BLACK)
                    if (board_stack[brd_level+1].
                        get_black_control_address()[i]>0)
                       value+=10; // bonus for checking king
            }
            if (b[i] == BLACK * KING) {
                if (side_moving == BLACK)
                    if (board_stack[brd_level+1].
                        get_white_control_address()[i]>0)
                       value-=1000; // king in check
                if (side_moving == WHITE)
                    if (board_stack[brd_level+1].
                        get_white_control_address()[i]>0)
                       value+=10; // bonus for checking king
            }
#ifdef TEXT_MODE
#ifdef DEBUG
            cerr << "** From: " 
                 << board_stack[brd_level].
                        move_from[possible_move_index]
                 << ", to: "
                 << board_stack[brd_level].
                        move_to[possible_move_index]
                 << ", i=" << i << ", wcontrol=" 
                 << board_stack[brd_level+1].wcontrol[i]
                 << ", bcontrol=" 
                 << board_stack[brd_level+1].bcontrol[i]
                 << ", value=" << value << "\n";
            cerr << "++ board_stack[brd_level+1].wcontrol[35]= "
                 << board_stack[brd_level+1].wcontrol[35] << "\n";
#endif
#endif
        }
    }

    int relative_control =
        board_stack[brd_level+1].get_white_control_address()[to_square] -
        board_stack[brd_level+1].get_black_control_address()[to_square];

    if (side_moving == 1 &&
        relative_control < 0)
    {
       value -= 75*p_value[b[to_square]];
    }

    if (side_moving ==-1 &&
        relative_control > 0)
    {
       value -= 75*p_value[-b[to_square]]; // sign of b[]
                                           // is negative
    }    

#if 0
    if (side_moving==1)
        board_stack[brd_level+1].wcontrol[to_square] +=1;
     else
        board_stack[brd_level+1].bcontrol[to_square] +=1;
#endif      
    return value;
}


// Use a static evaluation function to impose a
// partial ordering on the moves:

static char *column_name[] = {"A","B","C","D","E",
                              "F","G","H","Z"};

void game::rank_moves(int level)
{
    for (int i=0;
         i<board_stack[level].num_possible_moves;
         i++)
    {
        board_stack[level].move_value[i] = 
                     static_evaluation(level, i);
    }

    // Bubble sort the top 'width' moves to the top
    // of the move list. (NOTE: a bubble sort is very
    // efficient only if the number of items being 
    // moved is less than 16).

    for (i=0; i<width; i++) 
    {
        int max = -9999;
        int max_index = 0;
        for (int j=i+1; 
             j<board_stack[level].num_possible_moves;
             j++)
        {
            if (board_stack[level].move_value[j] > max)
            {
                max = board_stack[level].move_value[j];
                max_index = j;
            }
        }
        int temp = board_stack[level].move_value[max_index];
        board_stack[level].move_value[max_index] = 
                           board_stack[level].move_value[i];
        board_stack[level].move_value[i] = temp;
        temp = board_stack[level].move_from[max_index];
        board_stack[level].move_from[max_index] = 
                           board_stack[level].move_from[i];
        board_stack[level].move_from[i] = temp;
        temp = board_stack[level].move_to[max_index];
        board_stack[level].move_to[max_index] = 
                           board_stack[level].move_to[i];
        board_stack[level].move_to[i] = temp;
    }
#ifdef TEXT_MODE
    int num_loop = width;
    if (num_loop > board_stack[level].num_possible_moves)
        num_loop = board_stack[level].num_possible_moves;
    for (i=0; i<num_loop; i++) {
        cerr << "LEVEL=" << level << " ";
        cerr << "From: " << board_stack[level].move_from[i]
             << ", to: " << board_stack[level].move_to[i];
        int levelX, row, col;
        index_to_coords(board_stack[level].move_from[i],
                        row, col, levelX);
        cerr << "  (" << (levelX+1) << column_name[col]
             << (row + 1) << " -> ";
        
        index_to_coords(board_stack[level].move_to[i],
                        row, col, levelX);
        cerr << (levelX+1) << column_name[col] << (row + 1) << ")"           
             << ", value= " << board_stack[level].move_value[i]
             << "\n";
    }
#endif      
    return;
}

int game::make_best_move(int level, int side_flag)
{
#ifdef TEXT_MODE
    cerr << "MAKE_BEST_MOVE: level=" << level
         << ", side_flag=" << side_flag << "\n";
#endif
    if (level >= depth)
    {
        board_stack[level].set_current_side(side_flag);
        board_stack[level].generate_moves(0);
        rank_moves(level);
        if (board_stack[level].num_possible_moves > 0) 
        {
            int to_square   = board_stack[level].move_to[0];
            int from_square = board_stack[level].move_from[0];
            board_stack[level].get_board()[to_square] =
                        board_stack[level].get_board()[from_square];
            board_stack[level].get_board()[from_square] = 0;
        }
        return board_stack[level].num_possible_moves;
    }  else {  // do a limited lookahead:
        board_stack[level].set_current_side(side_flag);
        board_stack[level].generate_moves(0);
        rank_moves(level);
        for (int w=0; w<width; w++) {
            int to = board_stack[level].move_to[w];
            int from = board_stack[level].move_from[w];
            int save = board_stack[level].get_board()[from];
#ifdef TEXT_MODE
            cerr << "looking ahead: from=" << from
                 << ", to: " << to <<  "\n";
#endif
            board_stack[level+1] = board_stack[level];
            board_stack[level+1].get_board()[to] =
                          board_stack[level+1].get_board()[from];
            board_stack[level+1].get_board()[from] = 0;
            board_stack[level+1].set_current_side(-side_flag);
            make_best_move(level+1, -side_flag);
            board_stack[level].move_value[w] -= 
                          board_stack[level + 1].move_value[0];
        }

        // sort the moves:

        for (int i=0; i<width; i++) 
        {
            int max = board_stack[level].move_value[i];
            int max_index = i;
            int width2 = width;
            if (width2>board_stack[level].num_possible_moves)
                width2=board_stack[level].num_possible_moves;
            for (int j=i+1; j<width2; j++) 
            {
                if (board_stack[level].move_value[j] > max) 
                {
                    max = board_stack[level].move_value[j];
                    max_index = j;
                }
            }
            int temp = board_stack[level].move_value[max_index];
            board_stack[level].move_value[max_index] = 
                               board_stack[level].move_value[i];
            board_stack[level].move_value[i] = temp;
            temp = board_stack[level].move_from[max_index];
            board_stack[level].move_from[max_index] = 
                               board_stack[level].move_from[i];
            board_stack[level].move_from[i] = temp;
            temp = board_stack[level].move_to[max_index];
            board_stack[level].move_to[max_index] = 
                               board_stack[level].move_to[i];
            board_stack[level].move_to[i] = temp;
        }
        if (level == 0)
        {  
            int to = board_stack[level].move_to[0];
            int from = board_stack[level].move_from[0];
            board_stack[level].get_board()[to] =
                          board_stack[level].get_board()[from];
            board_stack[level].get_board()[from] = 0;
#ifndef TEXT_MODE
            int levelX1, row1, col1, levelX2, row2, col2;
            index_to_coords(board_stack[0].move_from[0],
                            row1, col1, levelX1);
            index_to_coords(board_stack[0].move_to[0],
                            row2, col2, levelX2);
            char buf2[128];
            sprintf(buf2,
                    "Computer move: %d%s%d -> %d%s%d",
                    levelX1+1, column_name[col1], row1+1,
                    levelX2+1, column_name[col2], row2+1);
            TAppWindow *current_window = 
                       Application::get_TAppWindow();
            current_window->put_scrolling_text(buf2);
#endif
#ifdef TEXT_MODE            
            int levelX, row, col;
            index_to_coords(board_stack[0].move_from[0],
                            row, col, levelX);
            cerr << "\nMOVE:   (" << (levelX+1)
                 << column_name[col] << (row + 1) << " -> ";         
            index_to_coords(board_stack[0].move_to[0],
                            row, col, levelX);
            cerr << (levelX+1) << column_name[col]
                 << (row + 1) << ")" << ", value= "
                 << board_stack[level].move_value[i]
                 << "\n";
#endif
        }
    }
}


#ifdef TEXT_MODE
void main()
{
    game b(1,4);
    
    for (int loop=0; loop<70; loop++) {
        if (b.make_best_move(0, 1) == 0)  break;
        b.board_stack[0].print_board();
        if (b.make_best_move(0,-1) == 0)  break;
        b.board_stack[0].print_board();
    }   
    
    
//  b.board_stack[0].set_pieces();
//  b.board_stack[0].current_side = 1;
//  b.board_stack[0].brd[35] = 0; 
//  b.board_stack[0].brd[55] = BLACK*PAWN;
//  b.board_stack[0].print_board();
//  b.board_stack[0].generate_moves(0);
//  b.rank_moves(0);
    
//  cerr << "WCONTROL, BCONTROL:\n";
//  for (int i=22; i<=99; i++)
//      cerr << i << " " << b.board_stack[1].wcontrol[i]
//           << " " << b.board_stack[1].bcontrol[i] << "\n";
//  cerr << "\n";

//  b.board_stack[0].print_possible_moves();
}

#else  // GUI mode:

char my_text[100];
char blowup_level_name[50];

game g(1,3);
int *b; // utility pointer directly to 
        // lookahead level 0 game board

const int row_spacing = 16;
const int col_spacing = 16;
const int level_spacing = 138;

// See Figure 17.3:

int x_pawn_template[] = { 4,  4,  6,  6, 5, 5, 6, 9,
                         10, 10,  9,  9, 11, 11};
int y_pawn_template[] = {15, 13, 13, 10, 9, 8, 7, 7,
                          8,  9, 10, 13, 13, 15};

int x_knight_template[] = { 3,  3,  6,  6, 5, 4, 4, 5,
                            5, 6, 6, 7, 10, 11, 12, 12,
                           11, 10,  9,  9, 11, 12, 12};
int y_knight_template[] = {15, 13, 13, 11, 9, 8, 7, 6,
                            6, 6, 4, 6,  7,  7,  8,  9,
                           10, 10, 11, 12, 13, 13, 15};

int x_bishop_template[] = { 3,  3,  6, 7, 5, 7, 8, 10,
                            8,  9, 12, 12};
int y_bishop_template[] = {15, 13, 12, 6, 4, 3, 3,  4,
                            6, 12, 13, 15};

int x_rook_template[] = { 3,  3,  5, 5, 4, 4, 5, 6, 9,
                         10, 11, 11, 10, 10, 12, 12};
int y_rook_template[] = {15, 12, 12, 7, 7, 3, 3, 6, 6,
                          3,  3,  7,  7, 12, 12, 15};

int x_queen_template[] = { 2,  2,  6,  3, 7, 6, 4, 3, 6,
                           9, 12, 11, 9, 8, 12,  9, 13, 13};
int y_queen_template[] = {15, 12, 11, 10, 9, 7, 6, 3, 2, 2,
                           3,  6, 7, 9, 10, 11, 12, 15};

int x_king_template[] = { 2,  2,  7, 6, 5, 6, 7, 7, 5, 5,
                          7, 7, 8, 8, 10, 10, 8, 8, 9, 10,
                          9,  8, 13, 13};
int y_king_template[] = {15, 12, 12, 7, 6, 4, 4, 2, 2, 1,
                          1, 0, 0,  1,  1, 2, 2, 4, 4, 6,
                          7, 12, 12, 15};

int *x_templates[] = {x_pawn_template, x_knight_template, 
                      x_bishop_template, x_rook_template,
                      x_queen_template, x_king_template};
int *y_templates[] = {y_pawn_template, y_knight_template, 
                      y_bishop_template, y_rook_template,
                      y_queen_template, y_king_template};

int num_templates[] = {14, 23, 12, 16, 18, 24};

int blowup_level = 0;
int hilight_square_list[2];
int num_hilight_squares = 0;

int redraw_small_boards = 1;
int redraw_large_board = 1;
void TAppWindow::update_display()
{
    // Plot the 3 "mini-board" displays in the left third
    // of the window:
    if (redraw_small_boards != 0)
        for (int level=2; level>=0;  level--) 
        {
            for (int col=7; col>=0; col--) 
            {
                for (int row=7; row>=0; row--) 
                {
                    int index = get_index(row, col, level);
                    int x = 5 + col_spacing * col;
                    int y = 8 + level_spacing * (2 - level) +
                            row_spacing * (7 - row);
                    plot_line(x,
                              y,
                              x + col_spacing,
                              y);
                    plot_line(x + col_spacing,
                              y,
                              x + col_spacing,
                              y + row_spacing);
                    plot_line(x + col_spacing,
                              y + row_spacing,
                              x,
                              y + row_spacing);
                    plot_line(x,
                              y + row_spacing,
                              x,
                              y);
                    // Check for hilighted squares:
                    for (int k=0; k<num_hilight_squares; k++)
                        if (hilight_square_list[k] == index) 
                        {
                            plot_line(x+1,
                                      y+1,
                                      x + col_spacing-1,
                                      y+1);
                            plot_line(x + col_spacing-1,
                                      y+1,
                                      x + col_spacing-1,
                                      y + row_spacing-1);
                            plot_line(x + col_spacing-1,
                                      y + row_spacing-1, 
                                      x+1,
                                      y + row_spacing-1);
                            plot_line(x+1,
                                      y + row_spacing-1,
                                      x+1,
                                      y+1);
                        }
                    // Shade the dark squares:
                    int rr = row % 2;
                    int cc = col % 2;
                    if ((rr == 0 && cc == 0) || 
                        (rr == 1 && cc == 1)) 
                    {
                        plot_line(x+2, y+4, x+4, y+2);
                        plot_line(x+(row_spacing - 4),
                                  y+col_spacing-2,
                                  x+row_spacing-2, 
                                  y+(col_spacing-4));
                        if (b[index] == 0) 
                        {
                            plot_line(x+2, 
                                      y+col_spacing-2,
                                      x+row_spacing-2,
                                      y+2);
                            plot_line(x+2, y+7, x+7, y+2);
                            plot_line(x+(row_spacing - 7),
                                      y+col_spacing-2,
                                      x+row_spacing-2, 
                                      y+(col_spacing-7));
                            plot_line(x+2, y+10, x+10, y+2);
                            plot_line(x+(row_spacing - 10),
                                      y+col_spacing-2, 
                                      x+row_spacing-2,
                                      y+(col_spacing-10));
                        }  else 
                        {
                            if (b[index] >= -4 &&
                                b[index] <= 4) 
                            {
                                plot_line(x+row_spacing-6,
                                          y+2+4,
                                          x+row_spacing-2,
                                          y+2);
                                plot_line(x+2, y+7, x+7, y+2);
                                if (b[index]==-1 ||
                                    b[index]==1) 
                                {
                                    plot_line(x+2, y+10,
                                              x+10, y+2);
                                }
                            } else 
                            {
                                plot_line(x+row_spacing-4, 
                                          y+2+2,
                                          x+row_spacing-2,
                                          y+2);
                            }
                        }
                    }
                    if (b[index] != 0 && b[index] != 7) 
                    {
                        int ptype = b[index];
                        if (ptype < 0)  ptype = -ptype;
                        ptype -= 1;
                        if (ptype >= 0 && ptype < 7) 
                        {
                            int num = num_templates[ptype];
                            for (int i=0; i< (num-1); i++)
                                plot_line(x+x_templates[ptype][i],
                                          y+y_templates[ptype][i],
                                          x+x_templates[ptype][i+1],
                                          y+y_templates[ptype][i+1]);
                        }
                        if (b[index] == -1)
                        {
                            plot_line(x+9, y+13, x+6, y+7);
                            for (int m=5; m<11; m++)
                                plot_line(x+m, y+row_spacing,
                                          x+m, y+row_spacing-2);
                        }
                        if (b[index] == -2) {
                            plot_line(x+9, y+12, x+6, y+6);
                            for (int m=4; m<12; m++)
                                plot_line(x+m, y+row_spacing,
                                          x+m, y+row_spacing-2);
                        }
                        if (b[index] == -3)
                        {
                            plot_line(x+9, y+12, x+7, y+6);
                            plot_line(x+8, y+6, x+7, y+3);
                            for (int m=4; m<12; m++)
                                plot_line(x+m, y+row_spacing,
                                          x+m, y+row_spacing-2);
                        }
                        if (b[index] < -3)
                        {
                            int m;
                            if (b[index] == -4)
                                for (m=4; m<12; m++)
                                    plot_line(x+m,
                                              y+row_spacing, 
                                              x+m,
                                              y+row_spacing-3);
                             else
                                for (m=3; m<13; m++)
                                    plot_line(x+m,
                                              y+row_spacing,
                                              x+m,
                                              y+row_spacing-3);
                            if (b[index] == BLACK*ROOK) 
                            {
                                plot_line(x+7,y+12,x+6, y+7);
                                plot_line(x+9,y+12,x+8, y+7);
                                plot_line(x+5, y+6, x+10, y+6);
                            }
                            if (b[index] == BLACK*QUEEN) 
                            {
                                plot_line(x+7, y+6, x+5, y+4);
                                plot_line(x+9, y+6, x+7, y+4);
                            }
                            if (b[index] == BLACK*KING) 
                            {
                                plot_line(x+8, y+12, x+6, y+4);
                                plot_line(x+9, y+7, x+8, y+4);
                            }
                        }
                    }
                }
            }
        }

    // Plot the large board display:

    if (redraw_large_board!= 0)
        for (int col=7; col>=0; col--) 
        {
            for (int row=7; row>=0; row--) 
            {
                int index = get_index(row, col, blowup_level);
                int x = 20 + col_spacing*2 *col + 9 * col_spacing;
                int y = 8 + row_spacing*2 * (7 - row);
                plot_line(x, y, x + col_spacing*2, y);
                plot_line(x + col_spacing*2, y,
                          x + col_spacing*2, y + row_spacing*2);
                plot_line(x + col_spacing*2, y + row_spacing*2,
                          x, y + row_spacing*2);
                plot_line(x, y + row_spacing*2, x, y);

                // Shade the dark squares:
                int rr = row % 2;
                int cc = col % 2;
                if ((rr == 0 && cc == 0) || (rr == 1 && cc == 1)) 
                {
                    plot_line(x+2, y+4, x+4, y+2);
                    plot_line(x+(row_spacing*2 - 4),
                              y+col_spacing*2-2,
                              x+row_spacing*2-2,
                              y+(col_spacing*2-4));
                    if (b[index] == 0) 
                    {
                        plot_line(x+2, y+col_spacing*2-2,
                                  x+row_spacing*2-2, y+2);
                        plot_line(x+2, y+7, x+7, y+2);
                        plot_line(x+(row_spacing*2 - 7),
                                  y+col_spacing*2-2,
                                  x+row_spacing*2-2,
                                  y+(col_spacing*2-7));
                        plot_line(x+2, y+10, x+10, y+2);
                        plot_line(x+(row_spacing*2 - 10), 
                                  y+col_spacing*2-2,
                                  x+row_spacing*2-2,
                                  y+(col_spacing*2-10));
                        plot_line(x+2, y+13, x+13, y+2);
                        plot_line(x+(row_spacing*2 - 13),
                                  y+col_spacing*2-2,
                                  x+row_spacing*2-2,
                                  y+(col_spacing*2-13));
                        plot_line(x+2, y+17, x+17, y+2);
                        plot_line(x+(row_spacing*2 - 17),
                                  y+col_spacing*2-2,
                                  x+row_spacing*2-2,
                                  y+(col_spacing*2-17));
                        plot_line(x+2, y+21, x+21, y+2);
                        plot_line(x+(row_spacing*2 - 21),
                                  y+col_spacing*2-2,
                                  x+row_spacing*2-2,
                                  y+(col_spacing*2-21));
                        plot_line(x+2, y+25, x+25, y+2);
                        plot_line(x+(row_spacing*2 - 25),
                                  y+col_spacing*2-2,
                                  x+row_spacing*2-2,
                                  y+(col_spacing*2-25));
                    }  else
                    {
                        plot_line(x+(row_spacing*2 - 2)-8,
                                  y+2+4,
                                  x+(row_spacing*2 - 2)-4,
                                  y+2);
                        plot_line(x+(row_spacing*2 - 2) -4,
                                  y+2+8,
                                  x+(row_spacing*2 - 2),
                                  y+2+4);
                        if (b[index] >= -4 && b[index] <= 4)
                        {
                            plot_line(x+row_spacing*2-2-8,
                                      y+2+8,
                                      x+row_spacing*2-2,
                                      y+2);
                            plot_line(x+2, y+7, x+7, y+2);
                            plot_line(x+2, y+10, x+10, y+2);
                        } else 
                        {
                            plot_line(x+row_spacing*2-2-6,
                            y+2+6,
                            x+row_spacing*2-2,
                            y+2);
                        }
                    }
                }
                if (b[index] != 0 && b[index] != 7) 
                {
                    int ptype = b[index];
                    if (ptype < 0)  ptype = -ptype;
                    ptype -= 1;
                    if (ptype >= 0 && ptype < 7) 
                    {
                        int num = num_templates[ptype];
                        for (int i=0; i< (num-1); i++)
                            plot_line(x+x_templates[ptype][i]*2,
                                      y+y_templates[ptype][i]*2+1,
                                      x+x_templates[ptype][i+1]*2,
                                      y+y_templates[ptype][i+1]*2+1);
                    }
                    if (b[index] == -1)
                    {
                        plot_line(x+18, y+26, x+12, y+14);
                        for (int m=5; m<11; m++)
                            plot_line(x+m*2,
                                       y+row_spacing*2-1,
                                       x+m*2,
                                       y+row_spacing*2-4);
                    }
                    if (b[index] == -2)
                    {
                        plot_line(x+18, y+24, x+12, y+12);
                        for (int m=4; m<12; m++)
                            plot_line(x+m*2,
                                      y+row_spacing*2-1,
                                      x+m*2,
                                      y+row_spacing*2-4);
                    }
                    if (b[index] == -3)
                    {
                        plot_line(x+18, y+24, x+14, y+12);
                        plot_line(x+16, y+12, x+14, y+6);
                        for (int m=4; m<12; m++)
                            plot_line(x+m*2,
                            y+row_spacing*2-1,
                            x+m*2,
                            y+row_spacing*2-4);
                    }
                    if (b[index] < -3) 
                    {
                        int m;
                        if (b[index] == -4)
                            for (m=4; m<12; m++)
                                plot_line(x+m*2,
                                          y+row_spacing*2-1,
                                          x+m*2,
                                          y+row_spacing*2-6);
                         else
                            for (m=3; m<13; m++)
                                plot_line(x+m*2,
                                          y+row_spacing*2-1,
                                          x+m*2,
                                          y+row_spacing*2-6);
                        if (b[index] == BLACK*ROOK) 
                        {
                            plot_line(x+14,y+24,x+12, y+14);
                            plot_line(x+18,y+24,x+16, y+14);
                            plot_line(x+10, y+12, x+20, y+12);
                        }
                        if (b[index] == BLACK*QUEEN) 
                        {
                            plot_line(x+14, y+12, x+10, y+8);
                            plot_line(x+18, y+12, x+14, y+8);
                        }
                        if (b[index] == BLACK*KING) 
                        {
                            plot_line(x+16, y+24, x+12, y+8);
                            plot_line(x+18, y+14, x+16, y+8);
                        }
                    }
                }
            }
        }

    sprintf(blowup_level_name,
            "Large board display is level %d",
            blowup_level+1);
    plot_string(10 * row_spacing + 6,
                2* level_spacing +2,
                blowup_level_name);
    plot_string(10,
                3 * level_spacing + 12,
                my_text);
}

static int do_computer_move = 0;

void TAppWindow::mouse_up(int x, int y)
{
    int level = 2 - (y - 8)/level_spacing;
    int row = 7 - ((y - ((2 - level) * level_spacing) - 8)
                 / row_spacing);
    int col = (x - 5) / col_spacing;
    if (level>=0 && level<3 && row>=0 &&
        row<8 && col>=0 && col < 8)
    {
        // Legal square selected on one of the
        // three "mini-boards":
        int index = get_index(row, col, level);
        if (num_hilight_squares<2) {
            hilight_square_list[num_hilight_squares++] = index;
            if (num_hilight_squares == 1)
            {
                redraw_large_board = 0;
                update_display();
                redraw_large_board = 1;
            }
            if (num_hilight_squares == 2) 
            {
                // Human selected a move:
                b[hilight_square_list[1]] = 
                          b[hilight_square_list[0]];
                b[hilight_square_list[0]] = 0;
                num_hilight_squares = 0;
                sprintf(my_text,"CALCULATING.......");
                erase_rect(1, 9*col_spacing, 600, 1);
                erase_rect(1, 600, 2* level_spacing +6,
                           10 * col_spacing);
                redraw_display();
                update_display();
                // let the computer move:
#if 0
                // comment out since idle_proc does this:
                g.make_best_move(0,-1);
                int to = g.board_stack[0].move_to[0];
#endif
                sprintf(my_text,"COMPUTER'S MOVE...");
                erase_rect(1, 9*col_spacing, 600, 1);
                erase_rect(1,
                           2* level_spacing -2,
                           30*col_spacing,
                           2* level_spacing -2);
                redraw_display();
                do_computer_move = 1;
            }
        }
    }  else // click outside mini-boards:
    {
        if (level>=0 && level<3) 
        {
            // switch board levels for large display:
            blowup_level = level;
            // erase_rect(1, 9*col_spacing, 600, 1);
            erase_rect(1, 600, 2* level_spacing +6,
                       10*col_spacing);
            redraw_small_boards = 0;
            update_display();
            redraw_small_boards = 1;
        }
    }
}

void TAppWindow::mouse_down(int, int)
{
}

void TAppWindow::mouse_move(int, int)
{
}

void TAppWindow::idle_proc()
{
    if (do_computer_move != 0) {
       do_computer_move = 0;
       g.make_best_move(0,-1);
       sprintf(my_text,"PLAYER'S MOVE.....");
       erase_rect(1, 9*col_spacing, 600, 1);
       erase_rect(1, 2* level_spacing -2,
                  30*col_spacing, 2* level_spacing -2);
       redraw_display();
       update_display();
    }
}

void TAppWindow::do_menu_action(int item_number)
{
    if (item_number == 1) 
    {
        g.get_board_address(0)->set_pieces();
        for (int m=0; m<14; m++)
            put_scrolling_text("  ");
        sprintf(my_text,"  ");
        clear_display();
        update_display();
    }
    if (item_number == 2)  
    {
        char buf[128];
        choose_file_to_write("Choose an output game file:", buf);
        filebuf out_file;
        if (out_file.open(buf, output)!=0)
        {
            ostream out_stream(&out_file);
            for (int m=0; m<360; m++)
                out_stream << b[m] << "\n";
            out_file.close();
        } else show_info("Could not open file for writing!");
    }
    if (item_number == 3)
    {
        char buf[128];
        choose_file_to_read("Choose a game file:", "ch", buf);
        filebuf in_file;
        if (in_file.open(buf, input)!=0) 
        {
            istream in_stream(&in_file);
            for (int m=0; m<360; m++) in_stream >> b[m];
            in_file.close();
        } else show_info("Could not open file for reading!");

    }
    if (item_number == 4)
    {
        exit(0);
    }
}

static char *menu_items[] =
    {"New game (play white)",
     "Save current game position",
     "Reload old game position"};

INIT_PROGRAM("Chess", 3, menu_items)

   // anything can go here:
   my_text[0] = '\0';
   blowup_level_name[0] = '\0';
   b = g.get_board_address(0)->get_board();
   TAppWindow *current_window = Application::get_TAppWindow();
   current_window->init_scrolling_text(2* level_spacing +11,
                                       27 * row_spacing - 10,
                                       2* level_spacing +150,
                                       10 * row_spacing + 4);
   RUN_PROGRAM;
}

#endif
